/*
 This file is part of GNU Taler
 (C) 2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { j2s, TalerCorebankApiClient } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  useSharedTestkudosEnvironment,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

/**
 * Tests for management of known bank accounts in the wallet.
 */
export async function runKnownAccountsTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bank, exchange } =
    await useSharedTestkudosEnvironment(t);

  const bankClient = new TalerCorebankApiClient(bank.baseUrl);

  // Withdraw digital cash into the wallet.

  await withdrawViaBankV3(t, {
    walletClient,
    bankClient,
    exchange,
    amount: "TESTKUDOS:20",
  });

  const accts = await walletClient.call(
    WalletApiOperation.ListBankAccounts,
    {},
  );

  console.log(`accounts: ${j2s(accts)}`);

  t.assertDeepEqual(accts.accounts.length, 1);

  await walletClient.call(WalletApiOperation.AddBankAccount, {
    paytoUri: "payto://iban/FOOBAR",
    label: "Foo",
    currencies: ["EUR"],
  });

  {
    const accts2 = await walletClient.call(
      WalletApiOperation.ListBankAccounts,
      {},
    );
    console.log(`accounts after add: ${j2s(accts2)}`);
    t.assertDeepEqual(accts2.accounts.length, 2);
  }

  // Can use add to update
  const addResp1 = await walletClient.call(WalletApiOperation.AddBankAccount, {
    paytoUri: "payto://iban/FOOBAR",
    label: "Foo",
    currencies: ["CHF"],
  });

  {
    const accts2 = await walletClient.call(
      WalletApiOperation.ListBankAccounts,
      {},
    );
    console.log(`accounts after modify: ${j2s(accts2)}`);
    t.assertDeepEqual(accts2.accounts.length, 2);
    const e = accts2.accounts.find((x) => x.label == "Foo");
    t.assertDeepEqual(e?.currencies, ["CHF", "EUR"]);
  }

  // Test replacement

  await walletClient.call(WalletApiOperation.AddBankAccount, {
    replaceBankAccountId: addResp1.bankAccountId,
    paytoUri: "payto://iban/QUUX",
    label: "Foo",
    currencies: ["CHF"],
  });

  {
    const accts2 = await walletClient.call(
      WalletApiOperation.ListBankAccounts,
      {},
    );
    console.log(`accounts after replace: ${j2s(accts2)}`);
    t.assertDeepEqual(accts2.accounts.length, 2);
    const e = accts2.accounts.find((x) => x.label == "Foo");
    t.assertDeepEqual(e?.paytoUri, "payto://iban/QUUX");
  }
}

runKnownAccountsTest.suites = ["wallet"];
